<?php
/**
 * The plugin update init file
 *
 * @package Podcast_Player
 */

if ( ! class_exists( 'PP_Pro_Plugin_Updater' ) ) {

	// load our custom updater if it doesn't already exist .
	include PP_PRO_DIR . '/admin/class-pp-pro-plugin-updater.php';
}

/**
 * Intiate plugin updation script.
 *
 * @since 1.0.0
 */
function pp_pro_plugin_updater() {
	// retrieve our license key from the DB.
	$license_key    = trim( get_option( 'pp_pro_license_key' ) );
	$active_plugins = get_option( 'active_plugins' );
	foreach ( $active_plugins as $active_plugin ) {
		if ( false !== strpos( $active_plugin, 'pp-pro.php' ) ) {
			$plugin_dir_and_filename = $active_plugin;
			break;
		}
	}
	if ( ! isset( $plugin_dir_and_filename ) || empty( $plugin_dir_and_filename ) ) {
		return;
	}
	// setup the updater.
	$edd_updater = new PP_Pro_Plugin_Updater(
		PP_PRO_STORE_URL,
		$plugin_dir_and_filename,
		array(
			'version' => PP_PRO_VERSION, // current version number.
			'license' => $license_key, // license key (used get_option above to retrieve from DB).
			'item_id' => PP_PRO_ITEM_ID, // id of this product in EDD.
			'author'  => 'vedathemes', // author of this plugin.
			'url'     => home_url(),
		)
	);
}
add_action( 'admin_init', 'pp_pro_plugin_updater', 0 );

/**
 * PP Extension license options markup.
 *
 * @since 1.0.0
 */
function pp_pro_license_options() {
	$status    = get_option( 'pp_pro_license_status' );
	$activate  = get_option( 'pp_license_activation' );
	$license   = get_option( 'pp_pro_license_key' );
	$action    = 'activate' === $activate ? 'deactivate' : 'activate';
	$is_active = false;
	$l_text    = esc_html__( 'Activate License', 'pp-pro' );
	if ( $license && false !== $status && 'valid' === $status ) {
		$is_active = true;
		$l_text    = esc_html__( 'Deactivate License', 'pp-pro' );
	}
	?>
	<div class="wrap">
		<h3><?php esc_html_e( 'Plugin License Options', 'pp-pro' ); ?></h3>
		<form method="post" action="options.php">
			<?php settings_fields( 'pp_pro_license' ); ?>
			<div class="license-activation-form">
				<?php if ( $is_active ) : ?>
					<div class="activation-msg">
						<span style="color:green;"><?php esc_html_e( 'License Activated', 'pp-pro' ); ?></span>
					</div>
				<?php endif; ?>
				<div class="license-input" <?php echo $is_active ? 'style="display: none"' : ''; ?>>
					<input id="pp_pro_license_key" name="pp_pro_license_key" type="text" placeholder="<?php esc_html_e( 'Enter your license key', 'pp-pro' ); ?>" class="regular-text" value="<?php echo esc_html( $license ); ?>" />
					<input id="pp_license_activation" name="pp_license_activation" type="hidden" value="<?php echo esc_html( $action ); ?>" />
				</div>
			</div>
			<?php submit_button( $l_text ); ?>
		</form>
	</div>
	<?php
}

/**
 * Register actiation key option.
 *
 * @since 1.0.0
 */
function pp_pro_register_option() {

	// creates our settings in the options table.
	register_setting( 'pp_pro_license', 'pp_pro_license_key', 'pp_pro_sanitize_license' );
	register_setting( 'pp_pro_license', 'pp_license_activation', 'pp_pro_sanitize_activation' );
}
add_action( 'admin_init', 'pp_pro_register_option' );

/**
 * Sanitize license key option.
 *
 * @param str $new New value of license key option.
 *
 * @since 1.0.0
 */
function pp_pro_sanitize_license( $new ) {
	$old = get_option( 'pp_pro_license_key' );
	if ( $old && $old !== $new ) {
		delete_option( 'pp_pro_license_status' ); // new license has been entered, so must reactivate.
	}
	return $new;
}

/**
 * Sanitize license key option.
 *
 * @param str $new New value of license key option.
 *
 * @since 1.0.0
 */
function pp_pro_sanitize_activation( $new ) {
	return 'activate' === $new ? 'activate' : 'deactivate';
}

/**
 * Activate OR Deactivate the license key.
 *
 * @since 1.0.0
 */
function pp_pro_mod_license_activation() {
	$status    = get_option( 'pp_pro_license_status' );
	$activate  = get_option( 'pp_license_activation' );
	$license   = get_option( 'pp_pro_license_key' );
	$is_active = false;

	if ( ! $license ) {
		return;
	}

	if ( false !== $status && 'valid' === $status ) {
		$is_active = true;
	}

	if ( $is_active && 'deactivate' === $activate ) {
		pp_pro_deactivate_license();
	} elseif ( ! $is_active && 'activate' === $activate ) {
		pp_pro_activate_license();
	}
}
add_action( 'admin_init', 'pp_pro_mod_license_activation' );

/**
 * Activate the license key.
 *
 * @since 1.0.0
 */
function pp_pro_activate_license() {

	// Retrieve the license from the database.
	$license = trim( get_option( 'pp_pro_license_key' ) );

	if ( ! $license ) {
		return;
	}

	// Data to send in our API request.
	$api_params = array(
		'edd_action' => 'activate_license',
		'license'    => $license,
		'item_id'    => PP_PRO_ITEM_ID, // The ID of the item in EDD.
		'url'        => home_url(),
	);

	// Call the custom API.
	$response = wp_remote_post(
		PP_PRO_STORE_URL,
		array(
			'timeout'   => 15,
			'sslverify' => false,
			'body'      => $api_params,
		)
	);

	// Make sure the response came back okay.
	if ( is_wp_error( $response ) || 200 !== wp_remote_retrieve_response_code( $response ) ) {
		$message = ( is_wp_error( $response ) && $response->get_error_message() ) ? $response->get_error_message() : esc_html__( 'An error occurred, please try again.', 'pp-pro' );
	} else {
		$license_data = json_decode( wp_remote_retrieve_body( $response ) );
		if ( false === $license_data->success ) {
			switch ( $license_data->error ) {
				case 'expired':
					$message = sprintf(
						esc_html__( 'Your license key expired on %s.', 'pp-pro' ),
						date_i18n( get_option( 'date_format' ), strtotime( $license_data->expires, current_time( 'timestamp' ) ) )
					);
					break;
				case 'revoked':
					$message = esc_html__( 'Your license key has been disabled.', 'pp-pro' );
					break;
				case 'missing':
					$message = esc_html__( 'Invalid license.', 'pp-pro' );
					break;
				case 'invalid':
				case 'site_inactive':
					$message = esc_html__( 'Your license is not active for this URL.', 'pp-pro' );
					break;
				case 'item_name_mismatch':
					$message = sprintf( esc_html__( 'This appears to be an invalid license key for %s.', 'pp-pro' ), 'pp-pro' );
					break;
				case 'no_activations_left':
					$message = esc_html__( 'Your license key has reached its activation limit.', 'pp-pro' );
					break;
				default:
					$message = esc_html__( 'An error occurred, please try again.', 'pp-pro' );
					break;
			}
		}
	}
	// Check if anything passed on a message constituting a failure.
	if ( ! empty( $message ) ) {
		delete_option( 'pp_pro_license_key' );
		$base_url = admin_url( 'admin.php?page=pp-options' );
		$redirect = add_query_arg(
			array(
				'sl_activation' => 'false',
				'message'       => rawurlencode( $message ),
			),
			$base_url
		);
		wp_safe_redirect( $redirect );
		exit();
	}
	// $license_data->license will be either "valid" or "invalid".
	update_option( 'pp_pro_license_status', $license_data->license );
	wp_safe_redirect( admin_url( 'admin.php?page=pp-options' ) );
	exit();
}

/**
 * Deactivate the license key.
 *
 * @since 1.0.0
 */
function pp_pro_deactivate_license() {

	// Retrieve the license from the database.
	$license = trim( get_option( 'pp_pro_license_key' ) );

	if ( ! $license ) {
		return;
	}

	// Data to send in our API request.
	$api_params = array(
		'edd_action' => 'deactivate_license',
		'license'    => $license,
		'item_id'    => PP_PRO_ITEM_ID, // The ID of the item in EDD.
		'url'        => home_url(),
	);

	// Call the custom API.
	$response = wp_remote_post(
		PP_PRO_STORE_URL,
		array(
			'timeout'   => 15,
			'sslverify' => false,
			'body'      => $api_params,
		)
	);

	// Make sure the response came back okay.
	if ( is_wp_error( $response ) || 200 !== wp_remote_retrieve_response_code( $response ) ) {

		if ( is_wp_error( $response ) ) {
			$message = $response->get_error_message();
		} else {
			$message = esc_html__( 'An error occurred, please try again.', 'pp-pro' );
		}

		delete_option( 'pp_pro_license_key' );
		$base_url = admin_url( 'admin.php?page=pp-options' );
		$redirect = add_query_arg(
			array(
				'sl_activation' => 'false',
				'message'       => rawurlencode( $message ),
			),
			$base_url
		);

		wp_safe_redirect( $redirect );
		exit();
	}

	// Decode the license data.
	$license_data = json_decode( wp_remote_retrieve_body( $response ) );

	// $license_data->license will be either "deactivated" or "failed"
	if ( 'deactivated' === $license_data->license ) {
		delete_option( 'pp_pro_license_status' );
	} else {
		delete_option( 'pp_pro_license_status' );
	}

	wp_safe_redirect( admin_url( 'admin.php?page=pp-options' ) );
	exit();
}
