<?php
/**
 * Podcast player utility functions.
 *
 * @link       https://www.vedathemes.com
 * @since      3.3.0
 *
 * @package    Podcast_Player
 * @subpackage Podcast_Player/Helper
 */

namespace PP_Pro\Helper\Functions;

/**
 * Podcast player utility functions.
 *
 * @package    Podcast_Player
 * @subpackage Podcast_Player/Helper
 * @author     vedathemes <contact@vedathemes.com>
 */
class Getters {

	/**
	 * Constructor method.
	 *
	 * @since  3.3.0
	 */
	public function __construct() {}

	/**
	 * Get list of all registered post types.
	 *
	 * @return array
	 */
	public static function get_post_types() {

		// Default Post and Pages post types.
		$default = array( 'post' => esc_html__( 'Posts', 'pp-pro' ) );

		// Get the registered post types.
		$post_types = get_post_types(
			array(
				'public'            => true,
				'_builtin'          => false,
				'show_in_nav_menus' => true,
			),
			'objects'
		);
		$post_types = wp_list_pluck( $post_types, 'label', 'name' );
		$post_types = array_merge( $default, $post_types );

		return $post_types;
	}

	/**
	 * Get list of all fonts.
	 *
	 * @return array
	 */
	public static function get_fonts_list() {

		/**
		 * Podcast player custom fonts list.
		 *
		 * @since 3.3.0
		 *
		 * @param array $fonts Fonts list.
		 */
		return apply_filters( 'podcast_player_fonts', array( '' => esc_html__( 'Default', 'podcast-player' ) ) );
	}

	/**
	 * Get list of taxonomies.
	 *
	 * @return array
	 */
	public static function get_taxonomies() {

		// Default taxonomies.
		$default = array(
			''         => esc_html__( 'Ignore Taxonomy', 'pp-pro' ),
			'category' => esc_html__( 'Categories', 'pp-pro' ),
			'post_tag' => esc_html__( 'Tags', 'pp-pro' ),
		);

		// Get list of all registered taxonomies.
		$taxonomies = get_taxonomies(
			array(
				'public'   => true,
				'_builtin' => false,
			),
			'objects'
		);

		// Get 'select' options as value => label.
		$options = wp_list_pluck( $taxonomies, 'label', 'name' );
		$options = array_merge( $default, $options );

		return $options;
	}

	/**
	 * Get list of taxonomies.
	 *
	 * @param  WP_REST_Request $request Request data.
	 *
	 * @return array
	 */
	public static function get_object_taxonomies( $request ) {

		$taxs = array();
		if ( isset( $request['post_type'] ) ) {
			// Get list of all registered taxonomies.
			$taxs = get_object_taxonomies( sanitize_text_field( $request['post_type'] ), 'objects' );
		}

		if ( empty( $taxs ) ) {
			return $taxs;
		}

		// Get 'select' options as value => label.
		$taxonomies = wp_list_pluck( $taxs, 'label', 'name' );

		if ( isset( $taxonomies['post_format'] ) ) {
			unset( $taxonomies['post_format'] );
		}

		$taxonomies[''] = esc_html__( '- Ignore Taxonomy -', 'pp-pro' );

		return $taxonomies;
	}

	/**
	 * Get list of taxonomies.
	 *
	 * @param  WP_REST_Request $request Request data.
	 *
	 * @return array
	 */
	public static function get_terms( $request ) {

		$terms = array();
		if ( isset( $request['taxonomy'] ) ) {
			$terms = get_terms(
				array(
					'taxonomy'   => sanitize_text_field( $request['taxonomy'] ),
					'hide_empty' => true,
				)
			);
			if ( is_wp_error( $terms ) ) {
				$terms = array();
			}
		}

		if ( empty( $terms ) ) {
			return $terms;
		}

		// Get 'select' options as value => label.
		$termlist = wp_list_pluck( $terms, 'name', 'slug' );
		return $termlist;
	}

	/**
	 * Get thumbnail image sizes attribute.
	 *
	 * @param array $args Settings for current Podcast player instance.
	 *
	 * @since  3.3.0
	 */
	public static function get_thumb_image_sizes( $args ) {
		$style = $args['display-style'];
		$sizes = '';

		if ( 'lv2' === $style ) {
			$sizes = '60px';
		} else {
			$sizes = '(max-width: 640px) 100vw, 640px';
		}

		return apply_filters( 'podcast_player_image_sizes', $sizes, $args );
	}
}
