<?php
/**
 * Podcast player display class.
 *
 * @link       https://www.vedathemes.com
 * @since      1.0.0
 *
 * @package    Podcast_Player
 */

namespace PP_Pro\Inc\Audio;

use Podcast_Player\Frontend\Inc\Render;
use Podcast_Player\Frontend\Inc\Instance_Counter;
use Podcast_Player\Helper\Functions\Getters as Get_Fn;
use Podcast_Player\Helper\Functions\Validation as Validation_Fn;

/**
 * Display podcast player instance.
 *
 * @package    Podcast_Player
 * @author     vedathemes <contact@vedathemes.com>
 */
class Audio {

	/**
	 * Holds the instance of this class.
	 *
	 * @since  1.0.0
	 * @access protected
	 * @var    object
	 */
	protected static $instance = null;

	/**
	 * Holds podcast episodes script data for each Podcast instance.
	 *
	 * @since  1.2.0
	 * @access protected
	 * @var    array
	 */
	protected $script_data = array();

	/**
	 * Holds media src index keys.
	 *
	 * @since  4.4.0
	 * @access private
	 * @var    string
	 */
	private $mkey = 'pp_media_src_index';

	/**
	 * Constructor method.
	 *
	 * @since  1.0.0
	 */
	public function __construct() {}

	/**
	 * Display current podcast player.
	 *
	 * @since  1.0.0
	 *
	 * @param array $args Podcast display args.
	 */
	public function display_podcast( $args ) {
		$args['audiosrc'] = Get_Fn::get_valid_media_url( $args['audiosrc'] );
		$media_type       = Get_Fn::get_media_type( $args['audiosrc'] );

		// Return if valid media type has not been provided.
		if ( ! $args['audiosrc'] ) {
			if ( is_admin() || current_user_can( 'manage_options' ) ) {
				echo '<p><strong>' . esc_html__( 'Fetch Error:', 'pp-pro' ) . '</strong> ' . esc_html__( 'Valid media url has not beed provided.', 'pp-pro' ) . '</p>';
			}
			return;
		}

		// Save audio src against md5 key in the database.
		$this->save_audio_src( $args['audiosrc'] );

		$inst_class = Instance_Counter::get_instance();
		$number     = $inst_class->get();

		// Prepare args and feed items for further use.
		$args       = $this->prepare_args( $args );
		$feed_items = $this->prepare_feed_items( $args, $number );

		// Add script data for current podcast instance.
		$this->add_podcast_script_data( $feed_items, $args, $number );

		$feed_items = array_values( $feed_items );

		$this->render_podcast_player( array( '', '', '', $feed_items, $number, $args, $this->script_data ) );
	}

	/**
	 * Prepare feed episodes for current podcast player instance.
	 *
	 * @since 1.0.0
	 *
	 * @param array $args     Episode Args.
	 * @param int   $instance Podcast player instance.
	 * @return array
	 */
	public function prepare_feed_items( $args, $instance ) {
		$feed_items = array();
		$id         = 'ppe-' . $instance . '-1';

		$link = $args['audiolink'] ? esc_attr( esc_url( $args['audiolink'] ) ) : '';

		// Send md5 key instead of actual media URL.
		$src  = $args['audiosrc'] ? esc_attr( md5( $args['audiosrc'] ) ) : '';
		$link = $link ? $link : ( $src ? $src : '' );

		$feed_items[ $id ]['title']       = $args['audiotitle'] ? esc_html( $args['audiotitle'] ) : '';
		$feed_items[ $id ]['description'] = '';
		$feed_items[ $id ]['author']      = '';
		$feed_items[ $id ]['date']        = '';
		$feed_items[ $id ]['link']        = $link;
		$feed_items[ $id ]['src']         = $src;
		$feed_items[ $id ]['mediatype']   = '';
		$feed_items[ $id ]['featured']    = esc_attr( esc_url( $args['imgurl'] ) );
		$feed_items[ $id ]['fset']        = esc_attr( $args['imgset'] );

		if ( isset( $args['description'] ) && $args['description'] ) {
			$feed_items[ $id ]['description'] = esc_html( $args['description'] );
		}

		return apply_filters( 'podcast_player_feed_items', $feed_items, 'link' );
	}

	/**
	 * Prepare args for current podcast player instance.
	 *
	 * @since 1.0.0
	 *
	 * @param array $args Episode Args.
	 * @return array
	 */
	public function prepare_args( $args ) {
		$hide_download = $args['ahide-download'] ? 1 : 0;
		$hide_social   = $args['ahide-social'] ? 1 : 0;

		if ( 'shortcode' === $args['from'] ) {
			$hide_social   = $hide_social ? $hide_social : ( $args['hide-social'] ? 1 : 0 );
			$hide_download = $hide_download ? $hide_download : ( $args['hide-download'] ? 1 : 0 );
		}

		// Set cover image url to player args if not already set.
		list( $imgurl, $imgset ) = $this->get_podcast_cover_image( $args );
		$args['imgurl']          = $imgurl;
		$args['imgset']          = $imgset;

		$imp = array(
			'hide-search'   => 1,
			'hide-author'   => 1,
			'hide-content'  => 1,
			'hide-loadmore' => 1,
			'hide-download' => $hide_download,
			'hide-social'   => $hide_social,
		);

		$args = wp_parse_args( $imp, $args );

		return $args;
	}

	/**
	 * Get podcast cover image url and markup.
	 *
	 * @since 1.0.0
	 *
	 * @param arr $args  Settings for current podcast player instance.
	 * @return array
	 */
	public function get_podcast_cover_image( $args ) {
		// Get main feed cover image markup.
		$id     = ! empty( $args['image'] ) ? absint( $args['image'] ) : '';
		$url    = ! empty( $args['img_url'] ) ? wp_strip_all_tags( $args['img_url'] ) : '';
		$srcset = '';

		if ( ! $id && $url ) {
			$id = attachment_url_to_postid( $url );
		}

		if ( $id ) {
			$imgdata = Get_Fn::get_image_src_set( $id, 'medium_large' );
			if ( $imgdata['src'] ) {
				$url    = $imgdata['src'];
				$srcset = $imgdata['srcset'];
			}
		}

		return array( $url, $srcset );
	}

	/**
	 * Add episodes data of current podcast instance to script data array.
	 *
	 * @since 1.0.0
	 *
	 * @param array $script_data podcast data.
	 * @param array $args   Podcast settings.
	 * @param int   $counter Maximum number of episodes to be displayed.
	 */
	public function add_podcast_script_data( $script_data = array(), $args, $counter ) {
		$render_info = array(
			'rdata' => array(
				'from' => 'link',
			),
		);
		$amsg        = isset( $args['audio-msg'] ) ? $args['audio-msg'] : '';
		if ( $amsg && 'audio' === Get_Fn::get_media_type( $amsg ) ) {
			$render_info['rdata'] = array_merge(
				$render_info['rdata'],
				array(
					'audiomsg' => esc_url( $args['audio-msg'] ),
					'playfreq' => absint( $args['play-freq'] ),
					'msgstart' => esc_html( $args['msg-start'] ),
					'msgtime'  => array_map( 'absint', $args['msg-time'] ),
					'msgtext'  => esc_html( $args['msg-text'] ),
				)
			);
		}

		$this->script_data[ 'pp-podcast-' . $counter ] = array_merge( $script_data, $render_info );
	}

	/**
	 * Populate podcast player cdata.
	 *
	 * @since 1.0.0
	 *
	 * @param array $data Podcast data.
	 * @return array
	 */
	public function scripts_data( $data = array() ) {
		if ( ! isset( $data['ajax_info'] ) ) {
			$data['ajax_info'] = array(
				'ajaxurl'  => admin_url( 'admin-ajax.php' ),
				'security' => wp_create_nonce( 'podcast-player-ajax-nonce' ),
			);
		}

		$data = array_merge( $data, $this->script_data );
		return $data;
	}

	/**
	 * Save media URL to the database.
	 *
	 * @since  4.4.0
	 *
	 * @param string $src Audio URL.
	 */
	private function save_audio_src( $src ) {
		$option = get_option( $this->mkey );
		$key    = md5( $src );
		$src    = esc_url_raw( $src );

		if ( ! $option || ! is_array( $option ) ) {
			$option = array( $key => $src );
		} elseif ( ! isset( $option[ $key ] ) ) {
			$option[ $key ] = $src;
		} else {
			$option = false;
		}

		if ( $option ) {
			update_option( $this->mkey, $option, 'no' );
		}
	}

	/**
	 * Fetch podcast episodes for Ajax calls.
	 *
	 * @since 1.0.0
	 */
	public function fetch_media_url() {
		check_ajax_referer( 'podcast-player-ajax-nonce', 'security' );

		$src    = isset( $_POST['src'] ) ? wp_unslash( $_POST['src'] ) : '';
		$option = get_option( $this->mkey );

		if ( $src && Validation_Fn::is_valid_url( $src ) ) {
			echo wp_json_encode( esc_url( $src ) );
			wp_die();
		}

		if ( $src && $option && is_array( $option ) && isset( $option[ $src ] ) ) {
			echo wp_json_encode( esc_url( $option[ $src ] ) );
			wp_die();
		}

		echo wp_json_encode( '' );
		wp_die();
	}

	/**
	 * Display podcast episodes.
	 *
	 * @since 1.0.0
	 *
	 * @param array $props Podcast player display props.
	 */
	private function render_podcast_player( $props ) {
		new Render( $props );
	}

	/**
	 * Returns the instance of this class.
	 *
	 * @since  1.0.0
	 *
	 * @return object Instance of this class.
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}
}
