<?php
/**
 * Podcast player display class.
 *
 * @link       https://www.vedathemes.com
 * @since      1.0.0
 *
 * @package    Podcast_Player
 */

namespace PP_Pro\Inc\Misc;

use Podcast_Player\Helper\Functions\Validation as Validation_Fn;
use PP_Pro\Helper\Functions\Getters as PP_Get_Fn;
use Podcast_Player\Helper\Functions\Markup as Markup_Fn;
use Podcast_Player\Helper\Functions\Utility as Utility_Fn;
use Podcast_Player\Helper\Functions\Getters as Get_Fn;

/**
 * Display podcast player instance.
 *
 * @package    Podcast_Player
 * @author     vedathemes <contact@vedathemes.com>
 */
class Render_Modification {

	/**
	 * Holds the instance of this class.
	 *
	 * @since  3.3.0
	 * @access protected
	 * @var    object
	 */
	protected static $instance = null;

	/**
	 * Constructor method.
	 *
	 * @since  3.3.0
	 */
	public function __construct() {}

	/**
	 * Display single episode wrapper.
	 *
	 * @since  3.3.0
	 *
	 * @param array  $callbacks Array of callback functions.
	 * @param object $render   Render object instance.
	 */
	public function episodes_list_wrapper( $callbacks, $render ) {
		$style = isset( $render->args['display-style'] ) ? $render->args['display-style'] : false;
		if ( $style && 'legacy' !== $style && 'modern' !== $style ) {
			$callbacks['pod-entry'] = array( $this, 'episodes_list' );
		}
		return $callbacks;
	}

	/**
	 * Modify podcast player markup to include pro features.
	 *
	 * @since  3.3.0
	 *
	 * @param array  $callbacks Array of callback functions.
	 * @param object $render    Render object instance.
	 */
	public function pp_podcast_single( $callbacks, $render ) {
		$args = $render->args;
		$amsg = isset( $args['audio-msg'] ) ? $args['audio-msg'] : '';
		if ( $amsg && 'audio' === Get_Fn::get_media_type( $amsg ) ) {
			$callbacks['pp-podcast__player'] = array( $this, 'podcast_player' );
		}
		return $callbacks;
	}

	/**
	 * Display single episode wrapper.
	 *
	 * @since 3.3.0
	 *
	 * @param string $classname Identifier unique classname.
	 * @param object $render    Render object instance.
	 */
	public function episodes_list( $classname, $render ) {
		$markup = '';
		$args   = $render->args;
		$style  = $args['display-style'];

		// Return if there is only one episode in the list.
		if ( Validation_Fn::is_single_episode_layout( count( $render->items ), $style ) ) {
			return $markup;
		}

		$template = Markup_Fn::locate_template( 'list', 'entry', true );
		if ( $template ) {
			$items = array_slice( $render->items, 0, $args['number'] );
			foreach ( $items as $key => $item ) {
				$ppe_id   = $render->instance . '-' . ( $key + 1 );
				$featured = $this->get_item_featured( $item, $args );
				$excerpt  = $this->get_item_excerpt( $item, $args );
				$class    = 'post' === $args['fetch-method'] ? 'fetch-post-title' : 'fetch-feed-title';
				ob_start();
				include $template;
				$markup .= ob_get_clean();
			}
		}
		$markup = Markup_Fn::remove_breaks( $markup );
		return $markup;
	}

	/**
	 * Display modified podcast player to include pro feature.
	 *
	 * Add custom audio feature to the existing podcast player markup.
	 *
	 * @since 3.3.0
	 *
	 * @param string $classname Identifier unique classname.
	 * @param object $render    Render object instance.
	 */
	public function podcast_player( $classname, $render ) {
		$first_item = isset( $render->items[0] ) ? $render->items[0] : '';
		if ( ! $first_item ) {
			return '';
		}

		$src    = isset( $first_item['src'] ) ? $first_item['src'] : '';
		$player = Markup_Fn::get_player_markup( $src, $render->instance );

		$caudio = '';
		$amsg   = isset( $render->args['audio-msg'] ) ? $render->args['audio-msg'] : '';
		if ( $amsg && 'audio' === Get_Fn::get_media_type( $amsg ) ) {
			$caudio = Markup_Fn::get_player_markup( $amsg, $render->instance . '-amsg' );
			$caudio = $caudio ? sprintf( '<div class="pp-player__amsg">%s</div>', $caudio ) : '';
		}

		if ( $player ) {
			return sprintf(
				'<div class="%1$s"><div class="pp-player-episode">%2$s</div>%3$s</div>',
				esc_attr( $classname ),
				$player,
				$caudio
			);
		}
		return '';
	}

	/**
	 * Get item's featured image markup.
	 *
	 * @since  3.3.0
	 *
	 * @param array $item Item args.
	 * @param array $args Settings for podcast player render instance.
	 */
	public function get_item_featured( $item, $args ) {
		$style = $args['display-style'];

		// Return if style does not support featured images.
		if ( ! Validation_Fn::is_style_support( $style, array( 'thumbnail', 'playbtn' ), 'OR' ) ) {
			return '';
		}

		// If user choose not to display featured images.
		if (
			! Validation_Fn::is_style_support( $style, 'playbtn' )
			&&
			$args['hide-featured']
		) {
			return '';
		}

		// Get featured image display data.
		if ( ! isset( $item['featured'] ) || ! $item['featured'] || Validation_Fn::is_style_support( $style, 'playbtn' ) ) {
			$imgurl = '';
		} else {
			$imgurl = $item['featured'];
		}
		$imgset   = isset( $item['fset'] ) ? $item['fset'] : '';
		$imgsizes = PP_Get_Fn::get_thumb_image_sizes( $args );

		// Get image markup.
		$image_markup = '';
		if ( $imgurl ) {
			$image_markup = sprintf(
				'<div class="pod-entry__thumb"><img class="pod-entry__image" src="%s" srcset="%s" sizes="%s"></div>',
				esc_url( $imgurl ),
				esc_attr( $imgset ),
				esc_attr( $imgsizes )
			);
		}

		// Play pause button markup.
		$play_icon = sprintf(
			'<div class="pod-entry__play"><span class="ppjs__offscreen">%s</span>%s%s</div>',
			esc_html__( 'Episode play icon', 'podcast-player' ),
			Markup_Fn::get_icon( array( 'icon' => 'pp-play' ) ),
			Markup_Fn::get_icon( array( 'icon' => 'pp-pause' ) )
		);

		return sprintf( '<div class="pod-entry__featured">%s%s</div>', $play_icon, $image_markup );
	}

	/**
	 * Get item's featured image markup.
	 *
	 * @since  3.3.0
	 *
	 * @param array $item Item args.
	 * @param array $args Settings for podcast player render instance.
	 */
	public function get_item_excerpt( $item, $args ) {
		$style = $args['display-style'];

		/*
		 * Return if style does not support excerpts
		 * OR if user choose not to display excerpts
		 */
		if (
			! Validation_Fn::is_style_support( $style, 'excerpt' )
			||
			! $args['excerpt-length']
		) {
			return '';
		}

		// Return item description is not available.
		if ( ! isset( $item['description'] ) || ! $item['description'] ) {
			return '';
		}

		if ( isset( $args['excerpt-unit'] ) && $args['excerpt-unit'] ) {
			$excerpt = mb_substr( wp_strip_all_tags( $item['description'], true ), 0, $args['excerpt-length'], 'utf-8' );
		} else {
			$excerpt = wp_trim_words(
				wp_strip_all_tags( $item['description'], true ),
				$args['excerpt-length'],
				''
			);
		}

		$class = 'post' === $args['fetch-method'] ? 'fetch-post-title' : 'fetch-feed-title';
		$link  = sprintf( '<a class="%1$s" href="%2$s">%3$s</a>', esc_attr( $class ), esc_url( $item['link'] ), '[...]' );
		return sprintf( '<div class="pod-entry__excerpt">%s%s</div>', $excerpt, $link );
	}

	/**
	 * Add pro specific dynamic css for podcast player render instance.
	 *
	 * @since 3.3.0
	 *
	 * @param string $css    Dynamic CSS.
	 * @param object $render Render object instance.
	 */
	public function dynamic_css( $css, $render ) {
		$args = $render->args;
		$inst = absint( $render->instance );
		$id   = "#pp-podcast-{$inst}";
		$mod  = ".modal-{$inst}";
		$amod = ".aux-modal-{$inst}";

		// Player background color customization.
		if ( Validation_Fn::is_style_support( $args['display-style'], 'bgcolor' ) ) {
			if ( isset( $args['bgcolor'] ) && $args['bgcolor'] ) {
				$bgrgb = Utility_Fn::hex_to_rgb( $args['bgcolor'], true );
				$css  .= sprintf(
					'
					%1$s.playerview,
					%1$s.playerview .episode-single,
					%1$s.lv4 .pod-entry,
					.inline-view %3$s,
					.inline-view %5$s {
						background-color: %2$s!important;
					}
					%1$s.playerview .pp-player__amsg {
						background-color: rgba( %4$s, 0.95 )!important;
					}
					',
					$id,
					sanitize_hex_color( $args['bgcolor'] ),
					$mod,
					$bgrgb,
					$amod
				);
			}
		}

		// Font family customization css.
		if ( isset( $args['font-family'] ) && $args['font-family'] ) {
			$ff     = $args['font-family'];
			$fflist = PP_Get_Fn::get_fonts_list();
			if ( isset( $fflist[ $ff ] ) ) {
				$font = $fflist[ $ff ];
				$css .= sprintf(
					'
					%1$s,
					%1$s input,
					%1$s .episode-list__load-more,
					%2$s,
					.pp-modal-window %3$s,
					.pp-modal-window %3$s input,
					.pp-modal-window %3$s .episode-list__load-more {
						font-family: "%4$s",-apple-system,BlinkMacSystemFont,Segoe UI,Helvetica,Arial,sans-serif,Apple Color Emoji,Segoe UI Emoji,Segoe UI Symbol !important;
					}
					',
					$id,
					$mod,
					$amod,
					$font
				);
			}
		}
		return $css;
	}

	/**
	 * Podcast player wrapper classes.
	 *
	 * @param array $classes Podcast player wrapper classes.
	 * @param array $args    Podcast display instance settings.
	 *
	 * @since  1.0.0
	 */
	public function wrapper_classes( $classes, $args ) {
		$style = $args['display-style'];

		// Set contrast color to the custom background color.
		if (
			Validation_Fn::is_style_support( $style, 'bgcolor' )
			&&
			$args['bgcolor']
		) {
			$is_dark_contrast = Validation_Fn::is_dark_contrast( $args['bgcolor'] );
			if ( ! $is_dark_contrast ) {
				$classes[] = 'light-color';
			}
		}

		// Make podcast player text color light.
		if (
			Validation_Fn::is_style_support( $style, 'txtcolor' )
			&&
			$args['txtcolor']
		) {
			$classes[] = 'light-text';
		}

		return $classes;
	}

	/**
	 * Returns the instance of this class.
	 *
	 * @since  1.0.0
	 *
	 * @return object Instance of this class.
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}
}
