<?php
/**
 * Get Post Data from Database.
 *
 * @link       https://www.vedathemes.com
 * @since      1.0.0
 *
 * @package    Podcast_Player
 * @subpackage Podcast_Player/Helper
 */

namespace PP_Pro\Inc\Post;

use PP_Pro\Helper\Functions\Utility as PP_Utility_Fn;
use Podcast_Player\Helper\Functions\Getters as Get_Fn;
use Podcast_Player\Helper\Functions\Validation as Validation_Fn;
use Podcast_Player\Frontend\Inc\General;

/**
 * Get Post Data from Database.
 *
 * @package    Podcast_Player
 * @subpackage Podcast_Player/Helper
 * @author     vedathemes <contact@vedathemes.com>
 */
class Get_Post_Data {

	/**
	 * Holds item fetching and display args.
	 *
	 * @since  3.3.0
	 * @access private
	 * @var    array
	 */
	private $args = array();

	/**
	 * Holds valid items list.
	 *
	 * @since  3.3.0
	 * @access private
	 * @var    array
	 */
	private $items = array();

	/**
	 * Constructor method.
	 *
	 * @since  3.3.0
	 *
	 * @param array $args Item episode fetching and display args.
	 */
	public function __construct( $args ) {
		// Set Object Properties.
		$this->args = $args;
	}

	/**
	 * Init method.
	 *
	 * @since  3.3.0
	 */
	public function init() {

		// Get items data from DB.
		$items = $this->get_post_ids();
		if ( is_wp_error( $items ) ) {
			return $items;
		}

		// Get required number of items having a media.
		$valid = $this->get_valid_items( $items );
		if ( is_wp_error( $valid ) ) {
			return $valid;
		}

		/*
		 * Get array of required number of valid items.
		 * Revised items list removes all invalid items upto
		 * the 'limit' count.
		 */
		list( $valid_items, $revised_items ) = $valid;

		// Make valid items available to all methods of this class.
		$ids         = array_column( $valid_items, 'id' );
		$this->items = array_combine( $ids, $valid_items );

		// Add additional data fields for valid items.
		$this->get_items_data();

		return $this->prepare_overall_data( $revised_items );
	}

	/**
	 * Init method.
	 *
	 * @since  3.3.0
	 */
	public function search() {

		// Get items data from DB.
		$items = $this->get_post_ids();
		if ( is_wp_error( $items ) ) {
			return $items;
		}

		// Get items for search specific Ajax request.
		$valid = $this->get_search_results( $items );
		if ( is_wp_error( $valid ) ) {
			return $valid;
		}

		// Make valid items available to all methods of this class.
		$ids         = array_column( $valid, 'id' );
		$this->items = array_combine( $ids, $valid );

		$this->get_items_data();
		return $this->items;
	}

	/**
	 * Get itemss ids from database or from args.
	 *
	 * @since  3.3.0
	 */
	public function get_post_ids() {

		// If ids are available from previous query.
		if ( ! empty( $this->args['ids'] ) ) {
			return $this->args['ids'];
		}

		// User selected items ids.
		$elist = array_filter( $this->args['elist'] );
		if ( ! empty( $elist ) ) {
			return $elist;
		}

		// Get item ids from database.
		$query_args = $this->prepare_query_args();
		$query      = new \WP_Query( $query_args );

		// Return array of item ids if available.
		if ( $query->have_posts() ) {
			return $query->posts;
		}

		return new \WP_Error(
			'no-items-error',
			esc_html__( 'No items available.', 'podcast-player' )
		);
	}

	/**
	 * Get default values of query parameters.
	 *
	 * @since  3.3.0
	 */
	private function prepare_query_args() {
		$query_args = $this->default_query_args();

		// Get specific post-type to be fetched.
		if ( $this->args['post-type'] ) {
			$query_args['post_type'] = $this->args['post-type'];
		}

		// Get data sort orderby.
		if ( false !== strpos( $this->args['sortby'], 'title' ) ) {
			$query_args['orderby'] = 'title';
		} else {
			$query_args['orderby'] = 'date';
		}

		// Get data sort order.
		if ( false !== strpos( $this->args['sortby'], 'desc' ) ) {
			$query_args['order'] = 'DESC';
		} else {
			$query_args['order'] = 'ASC';
		}

		// Get taxonomy filter args.
		if ( $this->args['taxonomy'] && ! empty( $this->args['terms'] ) ) {
			$query_args['tax_query'] = array(
				array(
					'taxonomy' => $this->args['taxonomy'],
					'field'    => 'slug',
					'terms'    => $this->args['terms'],
				),
			);
		}

		// Let's get all posts from the query.
		$query_args['posts_per_page'] = -1;

		// Get filter by text string.
		if ( $this->args['filterby'] ) {
			$query_args['pp_in_title'] = sanitize_text_field( $this->args['filterby'] );
		}

		return $query_args;
	}

	/**
	 * Get default values of query parameters.
	 *
	 * @since  3.3.0
	 */
	private function default_query_args() {
		return apply_filters(
			'podcast_player_posts_query_args',
			array(
				'post_type'           => 'post',
				'post_status'         => 'publish',
				'ignore_sticky_posts' => true,
				'no_found_rows'       => true,
				'fields'              => 'ids',
			)
		);
	}

	/**
	 * Filter items having a media file..
	 *
	 * @since  3.3.0
	 *
	 * @param array $items Item id list.
	 */
	public function get_valid_items( $items ) {
		$limit  = isset( $this->args['limit'] ) ? $this->args['limit'] : 20;
		$valid  = array();
		$count  = 0;
		$lindex = count( $items ); // First set last item index to maximum.
		foreach ( $items as $index => $id ) {

			// Break if we already got required number of items.
			if ( -1 !== $limit && $count >= $limit ) {
				$lindex = $index;
				break;
			}

			// Item content must contain at least one valid audio or video url.
			$media = $this->get_media( $id );
			if ( $media ) {
				$valid[] = array(
					'id'        => $id,
					'src'       => $media[0],
					'mediatype' => $media[1],
				);
				$count++;
			}
		}

		// Return array of item ids if available.
		if ( ! empty( $valid ) ) {
			$valid_ids = array_column( $valid, 'id' );
			$items     = array_merge( $valid_ids, array_slice( $items, $lindex ) );
			return array( $valid, $items );
		}

		return new \WP_Error(
			'no-media-error',
			esc_html__( 'No items available with media.', 'podcast-player' )
		);
	}

	/**
	 * Get results for Ajax search query.
	 *
	 * @since  3.3.0
	 *
	 * @param array $items Item id list.
	 */
	public function get_search_results( $items ) {
		$term      = $this->args['searchby'];
		$displayed = isset( $this->args['displayed'] ) ? $this->args['displayed'] : 0;

		// Items already displayed on front-end.
		$displayed_items = array_slice( $items, 0, $displayed );
		$valid           = array();

		/*
		 * Get all displayed items where title has the search term.
		 * All these items are already shown by JavaScript on front-end as
		 * search results. No need to process them again. Therefore,
		 * removing them from further processing.
		 */
		foreach ( $displayed_items as $index => $id ) {
			$title = strtolower( wp_strip_all_tags( get_the_title( $id ) ) );
			$term  = strtolower( $term );
			if ( false !== strpos( $title, $term ) ) {
				unset( $items[ $index ] );
			}
		}

		foreach ( $items as $index => $id ) {

			// Item content must contain at least one valid audio or video url.
			$media = $this->get_media( $id );
			if ( ! $media ) {
				continue;
			}
			if ( $term ) {

				// Look for the search term in item title.
				$title       = wp_strip_all_tags( get_the_title( $id ) );
				$is_searched = PP_Utility_Fn::string_search( $title, $term );

				// If not in title, look for the term in item content.
				if ( ! $is_searched ) {
					$content     = strtolower(
						wp_strip_all_tags( get_the_content( null, false, $id ) )
					);
					$is_searched = strpos( $content, $term );
				}

				// If still not available, it is not a valid item.
				if ( ! $is_searched ) {
					continue;
				}
			}
			$valid[] = array(
				'id'        => $id,
				'src'       => $media[0],
				'mediatype' => $media[1],
			);
		}

		// Return array of item ids if available.
		if ( ! empty( $valid ) ) {
			return $valid;
		}

		return new \WP_Error(
			'no-search-error',
			esc_html__( 'No search results available for the term.', 'podcast-player' )
		);
	}

	/**
	 * Get enclosed audio or video media from the item.
	 *
	 * @since 1.0.0
	 *
	 * @param int $id Item ID.
	 */
	public function get_media( $id ) {

		$instance = General::get_instance();
		$hdata    = Get_Fn::get_plugin_option( 'hide_data' );
		$mdata    = get_post_meta( $id, 'pp_import_data', true );
		if ( $mdata ) {
			return array( $mdata['src'], $mdata['type'] );
		}

		// Check if powerpress media is available.
		if ( function_exists( 'powerpress_get_enclosure_data' ) ) {
			$mdata = powerpress_get_enclosure_data( $id );
			if ( ! empty( $mdata['url'] ) ) {
				$mtype = ! empty( $mdata['type'] ) ? $mdata['type'] : '';
				$type  = false !== strpos( $mtype, 'audio' ) ? 'audio' : ( false !== strpos( $mtype, 'video' ) ? 'video' : $type );
				if ( $type ) {
					return array( $mdata['url'], $type );
				}
			}
		}

		// Check if seriously simple podcasting media file is available.
		if ( defined( 'SSP_VERSION' ) ) {
			$mdata = get_post_meta( $id, 'audio_file', true );
			$mtype = Get_Fn::get_media_type( $mdata );
			if ( $mtype && 'audio' === $mtype ) {
				return array( $mdata, 'audio' );
			}
		}

		// Temporarily disable masking filter.
		if ( 'yes' === $hdata ) {
			remove_filter( 'podcast_player_mask_audio_url', array( $instance, 'mask_audio_url' ) );
		}

		$content = '';

		// Get post content from Elementor (if any).
		if (
			in_array(
				'elementor/elementor.php',
				apply_filters( 'active_plugins', get_option( 'active_plugins' ) ),
				true
			)
		) {
			if ( class_exists( '\Elementor\Plugin' ) ) {
				$inst  = \Elementor\Plugin::instance();
				$front = false;
				if ( property_exists( $inst, 'frontend' ) ) {
					$front = $inst->frontend;
				}
				if ( $front && method_exists( $front, 'get_builder_content_for_display' ) ) {
					$content = $inst->frontend->get_builder_content_for_display( $id );
				}
			}
		}

		if ( $content ) {
			$content = wptexturize( $content );
		} else {
			// We want to replace special characters into formatted entities.
			$content = wptexturize( get_the_content( null, false, $id ) );

			// Strip podcast player block.
			$content = $this->render_blocks( $content );

			// Strip podcast player shortcode.
			$content = $this->strip_ppshortcode( $content );

			// Filter shortcodes in the content through their hooks.
			$content = do_shortcode( $content );
		}

		// Enable masking filter again.
		if ( 'yes' === $hdata ) {
			add_filter( 'podcast_player_mask_audio_url', array( $instance, 'mask_audio_url' ) );
		}

		return $this->get_media_from_content( $content );
	}

	/**
	 * Fetch media from post content.
	 *
	 * @since 3.0.0
	 *
	 * @param string $content post content.
	 */
	public function get_media_from_content( $content ) {
		$media = false;
		$type  = false;

		// Get media from the content.
		$audio = get_media_embedded_in_content( $content, array( 'audio' ) );
		if ( ! empty( $audio ) ) {
			$media = $audio[0];
			$type  = 'audio';
		} else {
			$video = get_media_embedded_in_content( $content, array( 'video' ) );
			if ( ! empty( $video ) ) {
				$media = $video[0];
				$type  = 'video';
			}
		}

		if ( false === $media ) {
			return false;
		}

		$media_url = $this->fetch_media_url( $media, $type );
		if ( false === $media_url ) {
			return false;
		}

		return array( $media_url, $type );
	}

	/**
	 * Check if there is podcast player shortcode with 'post' fetch method.
	 *
	 * @since 1.0.0
	 *
	 * @param string $content post content.
	 */
	public function strip_ppshortcode( $content ) {
		if ( false === strpos( (string) $content, '[podcastplayer' ) ) {
			return $content;
		}

		$tag = 'podcastplayer';
		if ( shortcode_exists( $tag ) ) {
			preg_match_all( '/' . get_shortcode_regex() . '/', $content, $matches, PREG_SET_ORDER );
			if ( empty( $matches ) ) {
				return $content;
			}

			foreach ( $matches as $shortcode ) {
				if ( $tag === $shortcode[2] && isset( $shortcode[3] ) ) {
					$atts = shortcode_parse_atts( $shortcode[3] );
					if ( ! is_array( $atts ) ) {
						continue;
					}
					if ( isset( $atts['fetch_method'] ) && 'post' === $atts['fetch_method'] ) {
						$content = str_replace( $shortcode[0], '', $content );
					}
				}
			}
		}
		return $content;
	}

	/**
	 * Check if there is podcast player block with 'post' fetch method.
	 *
	 * @since 1.0.0
	 *
	 * @param string $content post content.
	 */
	public function render_blocks( $content ) {
		if ( false === strpos( (string) $content, '<!-- wp:' ) ) {
			return $content;
		}

		if ( ! function_exists( 'parse_blocks' ) ) {
			return $content;
		}

		$blocks = parse_blocks( $content );
		if ( empty( $blocks ) ) {
			return $content;
		}

		$output = '';
		foreach ( $blocks as $block ) {
			if ( 'podcast-player/podcast-player' === $block['blockName'] ) {
				if ( isset( $block['attrs'] ) && is_array( $block['attrs'] ) ) {
					$atts = $block['attrs'];
					if ( isset( $atts['fetchMethod'] ) && 'post' === $atts['fetchMethod'] ) {
						continue;
					}
				}
			}
			$output .= render_block( $block );
		}

		return $output;
	}

	/**
	 * Fetch Media URL from media html markup.
	 *
	 * @since 1.0.0
	 *
	 * @param string $media Media HTML markup.
	 * @param string $type Media type.
	 */
	public function fetch_media_url( $media, $type ) {
		$media = wptexturize( $media );
		if ( false !== strpos( $media, 'src="' ) ) {
			$src = preg_match( '/src="([^"]*)"/i', $media, $array );
			$src = is_array( $array ) && isset( $array[1] ) ? $array[1] : '';
			$src = wp_strip_all_tags( $src );
			$src = preg_replace( '/\?.*/', '', $src );
			if ( Validation_Fn::is_valid_url( $src ) ) {
				return wp_specialchars_decode( esc_url( $src ) );
			} elseif ( $src ) {
				$option = get_option( 'pp_media_src_index' );
				if ( $option && is_array( $option ) && isset( $option[ $src ] ) ) {
					return esc_url( $option[ $src ] );
				}
			}
		}
		return false;
	}

	/**
	 * Modify the WHERE clause of the query.
	 *
	 * @since 1.0.0
	 *
	 * @param string   $where The WHERE clause of the query.
	 * @param WP_query $wp_query WP_Query instance.
	 */
	public function modify_query_where( $where, $wp_query ) {
		global $wpdb;
		$title_str = $wp_query->get( 'pp_in_title' );
		if ( $title_str ) {
			$where .= ' AND ' . $wpdb->posts . '.post_title LIKE \'%' . esc_sql( $wpdb->esc_like( $title_str ) ) . '%\'';
		}
		return $where;
	}

	/**
	 * Get required fata field for each valid item.
	 *
	 * @since  3.3.0
	 */
	public function get_items_data() {

		// Get items ids for fetching data.
		$ids = array_column( $this->items, 'id' );

		// Get data fields for individual items.
		$data = array_map(
			array( $this, 'get_item_data' ),
			$ids
		);

		// Combine data fields with corresponding item ID.
		$data = array_combine( $ids, $data );

		foreach ( $this->items as $key => &$val ) {
			$val2 = $data[ $key ];
			$val += $val2;
		}

		/**
		 * Data fields for valid post items.
		 *
		 * @since 3.3.0
		 *
		 * @param array $this->items Post items data array.
		 */
		$this->items = apply_filters(
			'podcast_player_post_items_data',
			$this->items
		);
	}

	/**
	 * Get single item data from db.
	 *
	 * @since  1.0.0
	 *
	 * @param int $id Item ID.
	 */
	private function get_item_data( $id ) {
		// Get default field values.
		$title       = trim( convert_chars( wptexturize( get_the_title( $id ) ) ) );
		$link        = esc_attr( esc_url( get_permalink( $id ) ) );
		$date        = esc_html( get_the_date( '', $id ) );
		$description = '';
		$author      = '';
		$src         = '';
		$srcset      = '';

		// Conditionally get item description.
		if ( ! $this->args['hide-content'] ) {
			$description = $this->get_item_description( $id );
		}

		// Conditionally get item author.
		if ( ! $this->args['hide-author'] ) {
			$author = $this->get_item_author( $id );
		}

		// Conditionally get item author.
		if ( ! $this->args['hide-featured'] ) {
			list( $src, $srcset ) = $this->get_item_featured( $id );
		}

		return array(
			'title'       => $title,
			'link'        => $link,
			'date'        => $date,
			'description' => $description,
			'author'      => $author,
			'featured'    => $src,
			'fset'        => $srcset,
		);
	}

	/**
	 * Get item's description form db.
	 *
	 * @since  1.0.0
	 *
	 * @param int $id Item ID.
	 */
	private function get_item_description( $id ) {
		if ( $this->args['display-style'] && 'legacy' !== $this->args['display-style'] && 'modern' !== $this->args['display-style'] ) {
			$length = $this->args['excerpt-length'] ? $this->args['excerpt-length'] : 25;
			if ( has_excerpt( $id ) ) {
				$text = get_the_excerpt( $id );
			} else {
				$text = wp_strip_all_tags( strip_shortcodes( get_the_content( null, false, $id ) ) );
				$text = str_replace( ']]>', ']]&gt;', $text );
			}
			$text = wp_trim_words( wp_strip_all_tags( $text, true ), $length, '' );
			$text = '<div>' . $text . '</div>';
		} else {
			// We want to replace special characters into formatted entities.
			$content = wptexturize( get_the_content( null, false, $id ) );

			// Strip podcast player shortcode, if fetch method is 'post'.
			$content = $this->render_filtered_blocks( $content );

			// Strip podcast player shortcode, if fetch method is 'post'.
			$content = strip_shortcodes( $content );

			$text = '<div>' . $content . '</div>';
		}
		return $text;
	}

	/**
	 * Get item's author form db.
	 *
	 * @since  1.0.0
	 *
	 * @param int $id Item ID.
	 */
	private function get_item_author( $id ) {
		$post = get_post( $id );
		if ( ! $post ) {
			return '';
		}
		$author = get_the_author_meta( 'display_name', $post->post_author );
		return esc_html( $author );
	}

	/**
	 * Get item's featured image form db.
	 *
	 * @since  1.0.0
	 *
	 * @param int $id Item ID.
	 */
	private function get_item_featured( $id ) {
		$featured_id = get_post_thumbnail_id( $id );
		$featured_id = apply_filters( 'podcast_player_post_featured', $featured_id, $this->args );
		$src         = '';
		$srcset      = '';
		if ( ! $featured_id ) {
			if ( isset( $this->args['imgurl'] ) && $this->args['imgurl'] ) {
				$src    = $this->args['imgurl'];
				$srcset = isset( $this->args['imgset'] ) ? $this->args['imgset'] : '';
			}
		} else {
			$imgdata = Get_Fn::get_image_src_set( $featured_id, 'medium_large' );
			$src     = $imgdata['src'];
			$srcset  = $imgdata['srcset'];
		}
		return array( $src, $srcset );
	}

	/**
	 * Check if there is podcast player block with 'post' fetch method.
	 *
	 * @since 1.0.0
	 *
	 * @param string $content post content.
	 */
	private function render_filtered_blocks( $content ) {
		if ( false === strpos( (string) $content, '<!-- wp:' ) ) {
			return $content;
		}

		if ( ! function_exists( 'parse_blocks' ) ) {
			return $content;
		}

		$blocks = parse_blocks( $content );
		if ( empty( $blocks ) ) {
			return $content;
		}

		$output = '';
		foreach ( $blocks as $block ) {
			$restricted_blocks = apply_filters(
				'pp_post_fetch_restricted_content',
				array(
					'podcast-player/podcast-player',
					'core/audio',
					'core/video',
					'core/gallery',
					'core/cover',
					'core/html',
					'core-embed',
				)
			);
			if ( in_array( $block['blockName'], $restricted_blocks, true ) ) {
				continue;
			} else {
				if ( in_array( 'core-embed', $restricted_blocks, true ) && false !== strpos( $block['blockName'], 'core-embed' ) ) {
					continue;
				}
			}

			$output .= render_block( $block );
		}

		return $output;
	}

	/**
	 * Get overall items and header level data.
	 *
	 * @since  3.3.0
	 *
	 * @param array $ids    Array of all items id to be used by JS query.
	 */
	private function prepare_overall_data( $ids ) {
		return array(
			'items' => $this->items,
			'ids'   => $ids,
		);
	}
}
