<?php
/**
 * Podcast player display class.
 *
 * @link       https://www.vedathemes.com
 * @since      1.0.0
 *
 * @package    Podcast_Player
 */

namespace PP_Pro\Inc\Post;

use PP_Pro\Inc\Post\Get_Post_Data;
use Podcast_Player\Frontend\Inc\Render;
use Podcast_Player\Frontend\Inc\Instance_Counter;
use Podcast_Player\Helper\Functions\Getters as Get_Fn;

/**
 * Display podcast player instance.
 *
 * @package    Podcast_Player
 * @author     vedathemes <contact@vedathemes.com>
 */
class Post {

	/**
	 * Holds the instance of this class.
	 *
	 * @since  1.0.0
	 * @access protected
	 * @var    object
	 */
	protected static $instance = null;

	/**
	 * Holds podcast episodes script data for each Podcast instance.
	 *
	 * @since  1.2.0
	 * @access private
	 * @var    array
	 */
	protected $script_data = array();

	/**
	 * Constructor method.
	 *
	 * @since  1.0.0
	 */
	public function __construct() {}

	/**
	 * Display current podcast player.
	 *
	 * @since  1.0.0
	 *
	 * @param array $args Podcast display args.
	 */
	public function display_podcast( $args ) {

		// Set cover image url to player args if not already set.
		if ( ! isset( $args['imgurl'] ) ) {
			list( $imgurl, $imgset ) = $this->get_podcast_cover_image( $args );
			$args['imgurl']          = $imgurl;
			$args['imgset']          = $imgset;
		}
		$args = $this->prepare_args( $args, 'primary' );
		$obj  = new Get_Post_Data( $args );
		$feed = $obj->init();
		if ( is_wp_error( $feed ) ) {
			if ( is_admin() || current_user_can( 'manage_options' ) ) {
				echo '<p><strong>' . esc_html__( 'Fetch Error:', 'podcast-player' ) . '</strong> ' . esc_html( $feed->get_error_message() ) . '</p>';
			}
			return;
		}

		// Save remaining items ids for future use by JavaScript.
		$args['ids'] = isset( $feed['ids'] ) ? $feed['ids'] : array();

		$inst_class = Instance_Counter::get_instance();
		$number     = $inst_class->get();

		// Prepare feed items for further use.
		$feed_items = $this->prepare_feed_items( $feed['items'], $number, 0, $args );
		$loaded     = count( $feed_items );

		if ( ! $loaded ) {
			if ( is_admin() || current_user_can( 'manage_options' ) ) {
				printf( 'No episodes available for given options.', 'pp-pro' );
			}
			return;
		}

		list( $fitems, $floaded ) = apply_filters( 'podcast_player_data_protect', array( $feed_items, $loaded ) );

		// Add script data for current podcast instance.
		$this->add_podcast_script_data( $fitems, $floaded, $args, $number );
		$feed_items = array_values( $feed_items );

		$this->render_podcast_player(
			array(
				$args['podtitle'],
				$args['description'],
				'',
				$feed_items,
				$number,
				$args,
				$this->script_data,
			)
		);
	}

	/**
	 * Prepare data fetch and display args for further use.
	 *
	 * @since 3.3.0
	 *
	 * @param array  $args     Settings for current podcast player instance.
	 * @param string $calledby Called by main query or Load more / search query.
	 */
	public function prepare_args( $args, $calledby = '' ) {

		// Set remaining items to be checked (if not set).
		$args['ids'] = isset( $args['ids'] ) ? $args['ids'] : array();

		// We need twice of required items in primary display.
		$args['limit'] = 'primary' === $calledby ? 2 * $args['number'] : $args['number'];

		return $args;
	}

	/**
	 * Get podcast cover image url and markup.
	 *
	 * @since 1.0.0
	 *
	 * @param arr $args  Settings for current podcast player instance.
	 * @return array
	 */
	public function get_podcast_cover_image( $args ) {
		// Get main feed cover image markup.
		$id     = ! empty( $args['image'] ) ? absint( $args['image'] ) : '';
		$url    = ! empty( $args['img_url'] ) ? wp_strip_all_tags( $args['img_url'] ) : '';
		$srcset = '';

		if ( ! $id && $url ) {
			$id = attachment_url_to_postid( $url );
		}

		if ( $id ) {
			$imgdata = Get_Fn::get_image_src_set( $id, 'medium_large' );
			if ( $imgdata['src'] ) {
				$url    = $imgdata['src'];
				$srcset = $imgdata['srcset'];
			}
		}

		return array( $url, $srcset );
	}

	/**
	 * Prepare feed episodes for current podcast player instance.
	 *
	 * @since 1.0.0
	 *
	 * @param array $items       Array of podcast episodes objects.
	 * @param int   $counter     Current podcast player instance number.
	 * @param int   $items_count Item number counter.
	 * @param array $args        Podcast player display args for this instance.
	 * @return array
	 */
	public function prepare_feed_items( $items, $counter, $items_count, $args ) {
		$feed_items = array();
		foreach ( $items as $key => $item ) {
			$id = $counter . '-' . ( ++$items_count );

			// Format feed items array to be used by JavaScript.
			$feed_items[ 'ppe-' . $id ] = $item;
		}
		return apply_filters( 'podcast_player_feed_items', $feed_items, 'post' );
	}

	/**
	 * Add episodes data of current podcast instance to script data array.
	 *
	 * @since 1.0.0
	 *
	 * @param array $feed_items Podcast data.
	 * @param int   $loaded Episodes loaded from query.
	 * @param array $args podcast player args for current instance.
	 * @param int   $counter Maximum number of episodes to be displayed.
	 */
	public function add_podcast_script_data( $feed_items = array(), $loaded, $args, $counter ) {
		$ajax_args = array(
			'number'  => absint( $args['number'] ),
			'display' => esc_html( $args['display-style'] ),
			'hdfeat'  => esc_html( $args['hide-featured'] ),
			'hddesc'  => esc_html( $args['hide-content'] ),
			'hdauth'  => esc_html( $args['hide-author'] ),
			'imgurl'  => esc_html( $args['imgurl'] ),
			'elength' => absint( $args['excerpt-length'] ),
		);

		$ajax_info = array(
			'load_info' => array(
				'displayed' => absint( $args['number'] ),
				'loaded'    => absint( $loaded ),
				'instance'  => absint( $counter ),
				'step'      => absint( $args['number'] ),
				'args'      => $ajax_args,
				'ids'       => $args['ids'] ? array_map( 'absint', $args['ids'] ) : array(),
			),
		);

		$render_info = array(
			'rdata' => array(
				'from'  => 'posts',
				'elen'  => absint( $args['excerpt-length'] ),
				'eunit' => isset( $args['excerpt-unit'] ) ? esc_html( $args['excerpt-unit'] ) : '',
				'title' => esc_html( $args['podtitle'] ),
			),
		);

		$amsg = isset( $args['audio-msg'] ) ? $args['audio-msg'] : '';
		if ( $amsg && 'audio' === Get_Fn::get_media_type( $amsg ) ) {
			$render_info['rdata'] = array_merge(
				$render_info['rdata'],
				array(
					'audiomsg' => esc_url( $args['audio-msg'] ),
					'playfreq' => absint( $args['play-freq'] ),
					'msgstart' => esc_html( $args['msg-start'] ),
					'msgtime'  => array_map( 'absint', $args['msg-time'] ),
					'msgtext'  => esc_html( $args['msg-text'] ),
				)
			);
		}

		$this->script_data[ 'pp-podcast-' . $counter ] = array_merge( $ajax_info, $feed_items, $render_info );
	}

	/**
	 * Display podcast episodes.
	 *
	 * @since 1.0.0
	 *
	 * @param array $props Podcast player display props.
	 */
	private function render_podcast_player( $props ) {
		new Render( $props );
	}

	/**
	 * Populate podcast player cdata.
	 *
	 * @since 1.0.0
	 *
	 * @param array $data Podcast data.
	 * @return array
	 */
	public function scripts_data( $data = array() ) {
		if ( ! isset( $data['ajax_info'] ) ) {
			$data['ajax_info'] = array(
				'ajaxurl'  => admin_url( 'admin-ajax.php' ),
				'security' => wp_create_nonce( 'podcast-player-ajax-nonce' ),
			);
		}

		$data = array_merge( $data, $this->script_data );
		return $data;
	}

	/**
	 * Fetch podcast episodes for Ajax calls.
	 *
	 * @since 1.0.0
	 */
	public function fetch_episodes() {
		check_ajax_referer( 'podcast-player-ajax-nonce', 'security' );

		$args     = isset( $_POST['args'] ) ? wp_unslash( $_POST['args'] ) : array();
		$instance = isset( $_POST['instance'] ) ? absint( wp_unslash( $_POST['instance'] ) ) : '';
		$loaded   = isset( $_POST['loaded'] ) ? absint( wp_unslash( $_POST['loaded'] ) ) : '';
		$ids      = isset( $_POST['ids'] ) ? array_map( 'absint', wp_unslash( $_POST['ids'] ) ) : array();

		// Return empty array if no more items left to check.
		if ( empty( $ids ) || $loaded >= count( $ids ) ) {
			echo wp_json_encode( array() );
			wp_die();
		}

		// Get ids for next set of iteration.
		$loaded_items = array_slice( $ids, 0, $loaded );
		$new_ids      = array_slice( $ids, $loaded );

		// Prepare args to fetch required items.
		$args = $this->prepare_args(
			array(
				'number'         => absint( $args['number'] ),
				'ids'            => $new_ids,
				'hide-content'   => $args['hddesc'] ? 1 : 0,
				'hide-author'    => $args['hdauth'] ? 1 : 0,
				'hide-featured'  => $args['hdfeat'] ? 1 : 0,
				'display-style'  => sanitize_text_field( $args['display'] ),
				'excerpt-length' => absint( $args['elength'] ),
			)
		);

		// Get next set of valid items.
		$obj  = new Get_Post_Data( $args );
		$feed = $obj->init();
		if ( is_wp_error( $feed ) ) {
			echo wp_json_encode( array() );
			wp_die();
		}

		// Prepare feed items for further use.
		$feed_items = $this->prepare_feed_items( $feed['items'], $instance, $loaded, $args );
		$count      = count( $feed_items );
		$loaded     = $loaded + $count;
		if ( $count ) {
			$output = array(
				'ids'      => array_merge( $loaded_items, $feed['ids'] ),
				'loaded'   => $loaded,
				'episodes' => $feed_items,
			);
			echo wp_json_encode( $output );
		} else {
			echo wp_json_encode( array() );
		}

		wp_die();
	}

	/**
	 * Fetch podcast episodes for Ajax calls.
	 *
	 * @since 1.0.0
	 */
	public function search_episodes() {
		check_ajax_referer( 'podcast-player-ajax-nonce', 'security' );

		$args      = isset( $_POST['args'] ) ? wp_unslash( $_POST['args'] ) : array();
		$ids       = isset( $_POST['ids'] ) ? array_map( 'absint', wp_unslash( $_POST['ids'] ) ) : '';
		$displayed = isset( $_POST['displayed'] ) ? absint( wp_unslash( $_POST['displayed'] ) ) : 0;
		$term      = isset( $_POST['search'] ) ? sanitize_text_field( wp_unslash( $_POST['search'] ) ) : false;

		// Return empty array if no more items left to check.
		if ( empty( $ids ) ) {
			echo wp_json_encode( array() );
			wp_die();
		}

		// Return if proper search term not provided.
		if ( ! $term || ! ( str_replace( ' ', '', $term ) ) ) {
			echo wp_json_encode( array() );
			wp_die();
		}

		$args = $this->prepare_args(
			array(
				'number'         => absint( $args['number'] ),
				'ids'            => $ids,
				'hide-content'   => $args['hddesc'] ? 1 : 0,
				'hide-author'    => $args['hdauth'] ? 1 : 0,
				'hide-featured'  => $args['hdfeat'] ? 1 : 0,
				'display-style'  => sanitize_text_field( $args['display'] ),
				'excerpt-length' => absint( $args['elength'] ),
				'searchby'       => $term,
				'displayed'      => $displayed,
			),
			'search'
		);

		// Get next set of valid items.
		$obj   = new Get_Post_Data( $args );
		$items = $obj->search();
		if ( is_wp_error( $items ) ) {
			echo wp_json_encode( array() );
			wp_die();
		}

		// Prepare feed items for further use.
		$feed_items = $this->prepare_feed_items( $items, 's', 0, $args );
		$count      = count( $feed_items );
		if ( $count ) {
			// Ajax output to be returened.
			$output = array( 'episodes' => $feed_items );
			echo wp_json_encode( $output );
		} else {
			echo wp_json_encode( array() );
		}

		wp_die();
	}

	/**
	 * Create REST API endpoints to get all pages list.
	 *
	 * @since 1.0.0
	 */
	public function register_routes() {
		register_rest_route(
			'podcastplayer/v1',
			'/pElist',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'pepisodes_list' ),
				'permission_callback' => function () {
					return current_user_can( 'edit_posts' );
				},
				'args'                => array(
					'postType' => array(
						'description' => esc_html__( 'Post Type', 'podcast-player' ),
						'type'        => 'string',
					),
					'sortBy'   => array(
						'description' => esc_html__( 'Sort by', 'podcast-player' ),
						'type'        => 'string',
					),
					'filterBy' => array(
						'description' => esc_html__( 'Filter by', 'podcast-player' ),
						'type'        => 'string',
					),
					'taxonomy' => array(
						'description' => esc_html__( 'Taxonomy', 'podcast-player' ),
						'type'        => 'string',
					),
					'terms'    => array(
						'description' => esc_html__( 'Terms', 'podcast-player' ),
						'type'        => 'string',
					),
				),
			)
		);

		register_rest_route(
			'podcastplayer/v1',
			'posttypes',
			array(
				'methods'             => 'GET',
				'callback'            => array( '\PP_Pro\Helper\Functions\Getters', 'get_post_types' ),
				'permission_callback' => function () {
					return current_user_can( 'edit_posts' );
				},
			)
		);

		register_rest_route(
			'podcastplayer/v1',
			'/taxonomies/(?P<post_type>[\w-]+)',
			array(
				'methods'             => 'GET',
				'callback'            => array( '\PP_Pro\Helper\Functions\Getters', 'get_object_taxonomies' ),
				'permission_callback' => function () {
					return current_user_can( 'edit_posts' );
				},
				'args'                => array(
					'post_type' => array(
						'description' => esc_html__( 'Post Type', 'pp-pro' ),
						'type'        => 'string',
					),
				),
			)
		);

		register_rest_route(
			'podcastplayer/v1',
			'/terms/(?P<taxonomy>[\w-]+)',
			array(
				'methods'             => 'GET',
				'callback'            => array( '\PP_Pro\Helper\Functions\Getters', 'get_terms' ),
				'permission_callback' => function () {
					return current_user_can( 'edit_posts' );
				},
				'args'                => array(
					'taxonomy' => array(
						'description' => esc_html__( 'Taxonomy', 'pp-pro' ),
						'type'        => 'string',
					),
				),
			)
		);

		register_rest_route(
			'podcastplayer/v1',
			'fontfamily',
			array(
				'methods'             => 'GET',
				'callback'            => array( '\PP_Pro\Helper\Functions\Getters', 'get_fonts_list' ),
				'permission_callback' => function () {
					return current_user_can( 'edit_posts' );
				},
			)
		);
	}

	/**
	 * Fetch podcast episodes list for admin Ajax calls.
	 *
	 * @param  WP_REST_Request $request Request data.
	 *
	 * @since 1.0.0
	 */
	public function pepisodes_list( $request ) {
		// Get variable values from Ajax request.
		$pt  = isset( $request['postType'] ) ? sanitize_text_field( $request['postType'] ) : 'post';
		$tax = isset( $request['taxonomy'] ) ? sanitize_text_field( $request['taxonomy'] ) : '';
		$ter = isset( $request['terms'] ) ? sanitize_text_field( $request['terms'] ) : '';
		$sby = isset( $request['sortby'] ) ? sanitize_text_field( $request['sortby'] ) : 'sort_date_desc';
		$fby = isset( $request['filterby'] ) ? sanitize_text_field( $request['filterby'] ) : '';
		$ter = array_filter( array_map( 'trim', explode( ',', $ter ) ) );

		$options = array( '' => esc_html__( 'Show All Episodes', 'podcast-player' ) );
		$sets    = array(
			'post-type' => $pt,
			'taxonomy'  => $tax,
			'terms'     => $ter,
			'sortby'    => $sby,
			'filterby'  => $fby,
			'number'    => -1,
		);

		// Get podcast episodes list.
		$list = $this->get_episodes_list( $sets );

		if ( isset( $list['episodes'] ) ) {
			return array_merge( $options, $list['episodes'] );
		} else {
			return array();
		}
	}

	/**
	 * Fetch podcast episodes list for admin Ajax calls.
	 *
	 * @since 1.0.0
	 */
	public function episodes_list() {
		check_ajax_referer( 'podcast-player-admin-ajax-nonce', 'security' );

		// Get variable values from Ajax request.
		$pt  = isset( $_POST['postType'] ) ? sanitize_text_field( wp_unslash( $_POST['postType'] ) ) : 'post';
		$tax = isset( $_POST['taxonomy'] ) ? sanitize_text_field( wp_unslash( $_POST['taxonomy'] ) ) : '';
		$ter = isset( $_POST['terms'] ) ? array_map( 'sanitize_text_field', wp_unslash( $_POST['terms'] ) ) : array();
		$sby = isset( $_POST['sortby'] ) ? sanitize_text_field( wp_unslash( $_POST['sortby'] ) ) : 'sort_date_desc';
		$fby = isset( $_POST['filterby'] ) ? sanitize_text_field( wp_unslash( $_POST['filterby'] ) ) : '';

		$sets = array(
			'post-type' => $pt,
			'taxonomy'  => $tax,
			'terms'     => $ter,
			'sortby'    => $sby,
			'filterby'  => $fby,
			'number'    => -1,
		);

		// Get podcast episodes list.
		$flist = $this->get_episodes_list( $sets );

		echo wp_json_encode( $flist );
		wp_die();
	}

	/**
	 * Fetch podcast episodes list for admin Ajax calls.
	 *
	 * @since 1.0.0
	 *
	 * @param arr $args Setting for current player instance.
	 */
	public function get_episodes_list( $args = array() ) {

		$default_args = array(
			'hide-content'  => 1,
			'hide-author'   => 1,
			'hide-featured' => 1,
			'elist'         => array(),
			'ids'           => array(),
			'limit'         => -1,
		);

		$args = wp_parse_args( $args, $default_args );

		// Get list of valid items.
		$obj  = new Get_Post_Data( $args );
		$feed = $obj->init();
		if ( is_wp_error( $feed ) ) {
			return array( 'error' => $feed->get_error_message() );
		}

		$items = $feed['items'];
		$ids   = array_column( $items, 'id' );
		$title = array_column( $items, 'title' );

		// Format items id and convert it to string.
		array_walk(
			$ids,
			function( &$val, $index ) {
				$val = '0' . $val;
			}
		);

		$fitems = array_combine( $ids, $title );
		$count  = count( $fitems );
		if ( $count ) {

			// Ajax output to be returened.
			return array(
				'loaded'   => $count,
				'episodes' => $fitems,
			);
		}
	}

	/**
	 * Modify the WHERE clause of the query.
	 *
	 * @since 1.0.0
	 *
	 * @param string   $where The WHERE clause of the query.
	 * @param WP_query $wp_query WP_Query instance.
	 */
	public function modify_query_where( $where, $wp_query ) {
		global $wpdb;
		$title_str = $wp_query->get( 'pp_in_title' );
		if ( $title_str ) {
			$where .= ' AND ' . $wpdb->posts . '.post_title LIKE \'%' . esc_sql( $wpdb->esc_like( $title_str ) ) . '%\'';
		}
		return $where;
	}

	/**
	 * Returns the instance of this class.
	 *
	 * @since  1.0.0
	 *
	 * @return object Instance of this class.
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}
}
